package com.ost.micro.provider.controller;
import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.form.EditForm;
import com.ost.micro.core.pay.modules.sys.dto.BizCompMapDto;
import com.ost.micro.core.pay.modules.sys.dto.BizSignCompExtDto;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.*;
import org.springframework.web.bind.annotation.*;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;

/**
 * @author Yubo
 */
@RestController
@RequestMapping("/mch/company")
@Api(tags = "签约公司接口")
@Slf4j
public class CompanyController {

    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ServiceUrl serviceUrl;

    private static final String BASE_URL = "/api/company";
    private static final String CASHIER_BASE_URL = "/cashier/company";


    @GetMapping("")
    @ApiOperation("查询公司列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "is_entry_info", value = "是否是资质录入员", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "商家id列表", required = false, dataType = "List"),
            @ApiImplicitParam(paramType = "query", name = "enable", value = "启用状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse search(@RequestParam(required = false) String keyword,
                               @RequestParam(required = false) Integer enable,
                               @RequestParam(required = false, name = "is_entry_info") Integer isEntryInfo,
                               @RequestParam(required = false, name = "biz_id") Long bizId,
                               @RequestParam(required = false, name = "biz_ids") String[] bizIds,
                               @RequestParam(name = "page_index") Integer pageIndex,
                               @RequestParam(name = "page_size") Integer pageSize) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("?keyword={keyword}");
        urlBuilder.append("&isEntryInfo={isEntryInfo}");
        urlBuilder.append("&bizId={bizId}");
        urlBuilder.append("&bizIds={bizIds}");
        urlBuilder.append("&enable={enable}");
        urlBuilder.append("&pageIndex={pageIndex}");
        urlBuilder.append("&pageSize={pageSize}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .getForEntity(urlBuilder.toString(), DataResponse.class, keyword, isEntryInfo, bizId, bizIds, enable, pageIndex, pageSize);

        return result.getBody();
    }

    @PostMapping("")
    @ApiOperation("添加签约公司")
    @DataToUnderline()
    //public DataResponse add(@RequestBody AddCompanyForm addCompanyForm) {
    public DataResponse add(@RequestBody BizSignCompExtDto addCompanyForm) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .postForEntity(urlBuilder.toString(), addCompanyForm, DataResponse.class);
        DataResponse dataResponse = result.getBody();
        return dataResponse;
    }

    @GetMapping("/info")
    @ApiOperation("获取签约公司信息")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "id", value = "公司id", required = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属商家集合", required = false, allowMultiple = true, dataType = "String")
    })
    @DataToUnderline()
    public DataResponse companyInfo(@RequestParam("id") Long id,
                                    @RequestParam(required = false, name = "biz_id") Long bizId,
                                    @RequestParam(required = false, name = "biz_ids") String[] bizIds) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/info?id={id}&bizId={bizId}&bizIds={bizIds}");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .getForEntity(urlBuilder.toString(), DataResponse.class, id, bizId, bizIds);
        return result.getBody();
    }

    @PutMapping("")
    @ApiOperation("编辑公司信息")
    @DataToUnderline()
    public DataResponse edit(@RequestBody EditForm editForm) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<EditForm> entity = new HttpEntity<EditForm>(editForm, headers);
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .exchange(urlBuilder.toString(), HttpMethod.PUT, entity, DataResponse.class);

        DataResponse dataResponse = result.getBody();
        return dataResponse;
    }

    @GetMapping("scope")
    @ApiOperation("获取经营范围列表")
    @DataToUnderline()
    public DataResponse scores() {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        urlBuilder.append("/scope");
        ResponseEntity<DataResponse> result = paymentServiceClient.getRestTemplate()
                .getForEntity(urlBuilder.toString(), DataResponse.class);
        return result.getBody();
    }

    @DeleteMapping("biz")
    @ApiOperation("删除公司对应的商家记录")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "comp_id", value = "公司id", required = true, dataType = "Long")})
    @DataToUnderline()
    public DataResponse compBizDelete(@RequestParam("biz_id") Long bizId,
                                      @RequestParam("comp_id") Long compId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        ResponseEntity<DataResponse> resp = paymentServiceClient.getRestTemplate().exchange(urlBuilder + "/biz?bizId={bizId}&compId={compId}", HttpMethod.DELETE, null, DataResponse.class, bizId, compId);
        return resp.getBody();
    }

    @PostMapping("/biz")
    @ApiOperation("添加公司对应的商家记录")
    @DataToUnderline()
    public DataResponse compBizAdd(@RequestBody BizCompMapDto dto) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        DataResponse dataResponse = paymentServiceClient.post(urlBuilder + "/biz", dto);
        return dataResponse;
    }

    @DeleteMapping("")
    @ApiOperation("删除签约公司")
    @ApiImplicitParam(paramType = "query", name = "comp_id", value = "公司id", required = true, dataType = "Long")
    @DataToUnderline()
    public DataResponse delCompany(@RequestParam("comp_id") Long compId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(serviceUrl.getAsServiceUrl(consulDiscoveryClient));
        urlBuilder.append(BASE_URL);
        ResponseEntity<DataResponse> resp = paymentServiceClient.getRestTemplate().exchange(urlBuilder + "?compId={compId}", HttpMethod.DELETE, null, DataResponse.class, compId);
        return resp.getBody();
    }

    @GetMapping("download")
    @ApiOperation("签约公司配置文件下载")
    public void download(@RequestParam(name = "cashier_url") String cashierUrl, HttpServletRequest request, HttpServletResponse response) {
        BufferedOutputStream buff = null;
        ServletOutputStream outSTr = null;
        try {
            ResourceLoader resourceLoader = new DefaultResourceLoader();
            Resource resource = resourceLoader.getResource("classpath:template/pay_cashier.conf");
            //Resource resource = new ClassPathResource("classpath:template/pay_cashier.conf");
            File file = resource.getFile();
            //File file = ResourceUtils.getFile("classpath:template/pay_cashier.conf")
            FileInputStream keyStoreIn = new FileInputStream(file);
            InputStreamReader in = new InputStreamReader(keyStoreIn, "UTF-8");
            BufferedReader br = new BufferedReader(in);
            StringBuffer content = new StringBuffer();
            String s = "";
            while ((s = br.readLine()) != null) {
                String line = String.format(s, cashierUrl);
                content.append(line).append(System.lineSeparator());
            }
            //使用占位符
            String outContext = String.format(content.toString(), cashierUrl);
            // 导出文件
            response.setContentType("text/plain");
            String fileName = "phone-list";
            response.setHeader("Content-Disposition", "attachment; filename=" + fileName + ".txt");
            outSTr = response.getOutputStream(); // 建立
            buff = new BufferedOutputStream(outSTr);
            // 把内容写入文件
            buff.write(outContext.getBytes("UTF-8"));
            buff.flush();
            buff.close();
        } catch (IOException e) {
            log.error("读取签约公司配置文件失败,error is {}", e);
        } finally {
            try {
                buff.close();
                outSTr.close();
            } catch (Exception e) {
                log.error("输入流关闭失败,error is {}", e);
            }
        }
    }

}
