package com.ost.micro.provider.controller;
import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.form.EditForm;
import com.ost.micro.core.pay.modules.sys.dto.BizAuditmpDto;
import com.ost.micro.core.pay.modules.sys.dto.BizDto;
import com.ost.micro.core.pay.modules.sys.dto.BizParamsDto;
import com.ost.micro.core.pay.modules.sys.dto.EditBizPsgDto;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.core.utils.RSAPublicKeyEncodeUtil;
import com.ost.micro.core.utils.ResponseUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

/**
 * 商家管理
 *
 * @author yubo
 */

@RestController
@Slf4j
@RequestMapping("/mch/biz")
@Api(tags = "商家管理接口")
public class BizController {

    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    private String bizUrl = null;// 商家
    private String certiUrl = null;// 商家交易详情
    private String orderUrl = null;// 商家交易详情
    private String psgUrl = null;// 商家设置支付通道
    private String psgGroupUrl = null;// 商家设置支付通道组
    private String transactionUrl = null;// 商家交易流水记录

    @GetMapping("")
    @ApiOperation("查询商家列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字，如：商家编号/法人姓名/手机号码/公司名称", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "audit_status", value = "资料认证状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "status", value = "启用状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "comp_id", value = "签约公司id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse searchBiz(@RequestParam(required = false, name = "keyword") String keyword,
                                  @RequestParam(required = false, name = "status") Integer status,
                                  @RequestParam(required = false, name = "audit_status") Integer auditStatus,
                                  @RequestParam(required = false, name = "comp_id") Long compId,
                                  @RequestParam(name = "page_index") Integer pageIndex,
                                  @RequestParam(name = "page_size") Integer pageSize) {
        if (bizUrl == null) {
            bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz";
        }
        Map params = new HashMap();
        params.put("keyword", keyword);
        params.put("auditStatus", auditStatus);
        params.put("status", status);
        params.put("compId", compId);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        return paymentServiceClient.get(bizUrl, params);
    }

    @PostMapping("")
    @ApiOperation("添加商家")
    @DataToUnderline()
    public DataResponse add(@RequestBody(required = false) BizDto dto) {
        if (bizUrl == null) {
            bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz";
        }
        return paymentServiceClient.post(bizUrl, dto);
    }

    @PutMapping("")
    @ApiOperation("编辑商家信息")
    @DataToUnderline()
    public DataResponse edit(@RequestBody EditForm bizForm) {
        if (bizUrl == null) {
            bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz";
        }
        paymentServiceClient.put(bizUrl, bizForm);
        return ResponseUtils.getInstance().success(null);
    }

    @GetMapping("{biz_id}")
    @ApiOperation("商家详情")
    @ApiImplicitParam(paramType = "path", name = "biz_id", value = "商家id", required = true, dataType = "Long")
    @DataToUnderline()
    public DataResponse bizDetail(@PathVariable("biz_id") Long bizId) {
        if (bizUrl == null) {
            bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz";
        }
        return paymentServiceClient.get(bizUrl, bizId);
    }

    @GetMapping("order/page")
    @ApiOperation("商家详情订单列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse bizDetailOrderList(@RequestParam(name = "biz_id") Long bizId,
                                           @RequestParam(name = "page_index") Integer pageIndex,
                                           @RequestParam(name = "page_size") Integer pageSize) {
        if (orderUrl == null) {
            orderUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/order/page";
        }
        Map<String, Object> params = new HashMap<>();
        params.put("bizId", bizId);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        return paymentServiceClient.get(orderUrl, params);
    }

    @GetMapping("psg")
    @ApiOperation("查询商家支付通道")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "商家id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "plat", value = "商家平台类型id", required = false, dataType = "long")})
    @DataToUnderline()
    public DataResponse getPassage(@RequestParam(name = "biz_id") Long bizId,
                                   @RequestParam Integer plat) {
        if (psgUrl == null) {
            psgUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/psg";
        }
        Map<String, Object> params = new HashMap<>();
        params.put("bizId", bizId);
        params.put("plat", plat);
        DataResponse dataResponse = paymentServiceClient.get(psgUrl, params);
        return dataResponse;
    }

    @PutMapping("psg")
    @ApiOperation("编辑商家支付通道")
    @DataToUnderline()
    public DataResponse editBizPassage(@RequestBody EditBizPsgDto editBizPsgForm) {
        if (psgUrl == null) {
            psgUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/psg";
        }
        paymentServiceClient.put(psgUrl, editBizPsgForm);
        return ResponseUtils.getInstance().success(null);
    }

    @PostMapping("psg")
    @ApiOperation("添加商家支付通道")
    @DataToUnderline()
    public DataResponse addBizPassage(@RequestBody EditBizPsgDto editBizPsgForm) {
        if (psgUrl == null) {
            psgUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/psg";
        }
        paymentServiceClient.post(psgUrl, editBizPsgForm);
        return ResponseUtils.getInstance().success(null);
    }

    @DeleteMapping("psg/{biz_id}")
    @ApiOperation("重置商家支付通道")
    @ApiImplicitParam(paramType = "path", name = "biz_id", value = "商家id", required = true, dataType = "Long")
    @DataToUnderline()
    public DataResponse restBizPassage(@PathVariable("biz_id") Long bizId) {
        if (psgUrl == null) {
            psgUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/psg";
        }
        Map<String, Object> params = new HashMap<>();
        params.put("bizId", bizId);
        paymentServiceClient.delete(psgUrl, bizId);
        return ResponseUtils.getInstance().success(null);
    }

    @GetMapping("psg/group/{biz_id}")
    @ApiOperation("获取支付通道组")
    @ApiImplicitParam(paramType = "path", name = "biz_id", value = "商家id", required = true, dataType = "Long")
    @DataToUnderline()
    public DataResponse getPsgGroup(@PathVariable("biz_id") Long bizId) {
        if (psgGroupUrl == null) {
            psgGroupUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/psg/group";
        }
        return paymentServiceClient.get(psgGroupUrl, bizId);
    }

    @PostMapping("certifile")
    @ApiOperation("添加认证资料")
    @DataToUnderline()
    public DataResponse addCertiFile(@RequestBody BizAuditmpDto dto) {
        if (certiUrl == null) {
            certiUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/certifile";
        }
        return paymentServiceClient.post(certiUrl, dto);
    }

    @GetMapping("certifile/{biz_id}")
    @ApiOperation("根据商家id获取认证资料")
    @ApiImplicitParam(paramType = "path", name = "biz_id", value = "商家id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse getCertiFile(@PathVariable("biz_id") Long bizId) {
        if (certiUrl == null) {
            certiUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/certifile";
        }
        return paymentServiceClient.get(certiUrl, bizId);
    }

    @PutMapping("certifile")
    @ApiOperation("编辑认证资料")
    @DataToUnderline()
    public DataResponse editCertiFile(@RequestBody EditForm editForm) {
        if (certiUrl == null) {
            certiUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/certifile";
        }
        paymentServiceClient.put(certiUrl, editForm);
        return ResponseUtils.getInstance().success(null);

    }

    @GetMapping("params/{biz_id}")
    @ApiOperation("获取交易参数")
    @ApiImplicitParam(paramType = "path", name = "biz_id", value = "商家id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse getParams(@PathVariable("biz_id") Long bizId) {
        String paramsUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/params";
        return paymentServiceClient.get(paramsUrl, bizId);
    }

    @PostMapping("/params")
    @ApiOperation("添加交易参数")
    @DataToUnderline()
    public DataResponse addParams(@RequestBody BizParamsDto bizParamsDto) {

        String paramsUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/params";
        //调用as-service服务
        DataResponse backResponse = paymentServiceClient.post(paramsUrl, bizParamsDto);
        log.info("添加交易参数 as-service 响应结果为:", GsonUtil.toJson(backResponse, true));
        return backResponse;
    }

    @PutMapping("params")
    @ApiOperation("编辑交易参数")
    @DataToUnderline()
    public DataResponse editParams(@RequestBody EditForm editForm) {

        //调用as-service服务
        String paramsUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/params";
        DataResponse backResponse = paymentServiceClient.putForEntity(paramsUrl, editForm);
        log.info("编辑交易参数 as-service 响应结果为:", GsonUtil.toJson(backResponse, true));
        return backResponse;
    }

    @GetMapping("transaction/{id}")
    @ApiOperation("获取交易详情")
    @ApiImplicitParam(paramType = "path", name = "id", value = "id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse getTradeDetail(@PathVariable Long id) {
        if (transactionUrl == null) {
            transactionUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/biz/transaction";
        }
        return paymentServiceClient.get(transactionUrl, id);
    }

    @GetMapping("/ras")
    @ApiOperation("获取公钥与私钥")
    @DataToUnderline()
    public DataResponse crateRas() {
        Map<String, String> dataMap = RSAPublicKeyEncodeUtil.generateKeyPair();
        return ResponseUtils.getInstance().success(dataMap);
    }

}
