package com.ost.micro.provider.common;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.LongSerializationPolicy;
import com.ost.micro.common.utils.Result;
import com.ost.micro.core.configuration.SpringfoxJsonToGsonAdapter;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.core.utils.MapUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.*;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.GsonHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Service;
import org.springframework.util.MimeType;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author david.zhong
 * @Title: PaymentServiceClient
 * @ProjectName micro-root
 * @Description: 远程调用底层逻辑
 * @date 2018/9/3011:03
 */
@Service
@Slf4j
public class PaymentServiceClient<T> {
    private Gson gson = new Gson();
    private RestTemplate restTemplate;

    public PaymentServiceClient() {
        restTemplate = new RestTemplate();
        List<HttpMessageConverter<?>> converters = restTemplate.getMessageConverters();
        // 删除MappingJackson2HttpMessageConverter
        converters.removeIf(httpMessageConverter -> httpMessageConverter instanceof MappingJackson2HttpMessageConverter);
        // 添加GsonHttpMessageConverter
        converters.add(new GsonHttpMessageConverter(new GsonBuilder().enableComplexMapKeySerialization().setLongSerializationPolicy(LongSerializationPolicy.STRING)
                .serializeNulls().registerTypeAdapter(JsonElement.class, new SpringfoxJsonToGsonAdapter()).create()));
    }

    /**
     * @param url
     * @param param
     * @return
     */
    // @HystrixCommand(fallbackMethod = "selectPayChannelFallback") 熔断机制
    public DataResponse get(String url, Map param) {
        String urlParams = MapUtil.getUrlParamsByMap(param);
        if (!StringUtils.isEmpty(urlParams)) {
            url = url + "?" + urlParams;
        }
        log.info("get url is :" + url);
        return restTemplate.getForEntity(url, DataResponse.class).getBody();
    }

    /**
     * 包含敏感信息（身份信息、权限信息），用此方法做脱敏处理
     *
     * @param url
     * @param param
     * @return
     */
    public Result _get(String url, Map param) {
        String urlParams = MapUtil.getUrlParamsByMap(param);
        if (!StringUtils.isEmpty(urlParams)) {
            url = url + "?" + urlParams;
        }
        log.info("get url is :" + url.split("[?]")[0] + "?****");
        return restTemplate.getForEntity(url, Result.class).getBody();
    }

    /**
     * 通过id去获取
     *
     * @param url
     * @param id
     * @return
     */
    public DataResponse get(String url, Long id) {
        url = url + "/" + id;
        log.info("get url is :" + url);
        DataResponse model = restTemplate.getForEntity(url, DataResponse.class).getBody();
        String json = gson.toJson(model.getData());
        Map<String, Object> data = gson.fromJson(json, HashMap.class);
        if (null != data) {
            data.put("id", id);
        }
        return model;
    }

    /**
     * post 请求
     *
     * @param url
     * @param t
     * @return
     */
    public DataResponse post(String url, T t) {
        log.info("post url is :" + url);
        log.info("request params is :" + gson.toJson(t));
        return restTemplate.postForEntity(url, t, DataResponse.class).getBody();
    }

    /**
     * post 请求
     *
     * @param url
     * @param t
     * @return
     */
    public DataResponse postForObject(String url, MediaType mediaType, T t) {
        log.info("post url is :" + url);
        log.info("request params is :" + gson.toJson(t));
        //return restTemplate.postForEntity(url, t, DataResponse.class).getBody();
        HttpHeaders header = new HttpHeaders();
        // 需求需要传参为application/json格式
        header.setContentType(mediaType);
        HttpEntity<String> httpEntity = new HttpEntity<>(GsonUtil.toJson(t, false), header);
        String response = restTemplate.postForObject(url, httpEntity, String.class);
        log.info("response is {}", response);
        DataResponse dataResponse = GsonUtil.fromJson(response, DataResponse.class);
        return dataResponse;
    }

    /**
     * put请求
     *
     * @param url
     * @param t
     * @return
     */
    public void put(String url, T t) {
        log.info("put url is :" + url);
        log.info("request params is :" + gson.toJson(t));
        restTemplate.put(url, t);
    }

    /**
     * put请求
     *
     * @param url
     * @param t
     * @return
     */
    public DataResponse putForEntity(String url, T t) {
        log.info("put url is :" + url);
        log.info("request params is :" + gson.toJson(t));
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<T> entity = new HttpEntity<T>(t, headers);
        ResponseEntity<DataResponse> result = restTemplate.exchange(url,
                HttpMethod.PUT, entity, DataResponse.class);
        return result.getBody();
    }

    /**
     * delete 请求
     *
     * @param url
     * @param id
     */
    public void delete(String url, Long id) {
        url = url + "/" + id;
        log.info("request params is :" + url);
        restTemplate.delete(url, id);
    }

    /**
     * delete
     *
     * @param url
     * @param t
     * @return
     */
    public DataResponse deleteForEntity(String url, T t) {
        log.info("delete url is :" + url);
        log.info("request params is :" + gson.toJson(t));
        //restTemplate.put(url, t);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<T> entity = new HttpEntity<T>(t, headers);
        ResponseEntity<DataResponse> result = restTemplate.exchange(url,
                HttpMethod.DELETE, entity, DataResponse.class, t);
        return result.getBody();
    }

    /**
     * 发送/获取 服务端数据(主要用于解决发送put,delete方法无返回值问题).
     *
     * @param url      绝对地址
     * @param method   请求方式
     * @param bodyType 返回类型
     * @param <T>      返回类型
     * @return 返回结果(响应体)
     */
    public <T> T exchange(String url, HttpMethod method, Class<T> bodyType, Map<String, Object> params) {
        // 请求头
        HttpHeaders headers = new HttpHeaders();
        MimeType mimeType = MimeTypeUtils.parseMimeType("application/json");
        MediaType mediaType = new MediaType(mimeType.getType(), mimeType.getSubtype(), Charset.forName("UTF-8"));
        // 请求体
        headers.setContentType(mediaType);
        //提供json转化功能
        ObjectMapper mapper = new ObjectMapper();
        String str = null;
        try {
            if (!params.isEmpty()) {
                str = mapper.writeValueAsString(params);
            }
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        // 发送请求
        HttpEntity<String> entity = new HttpEntity<>(str, headers);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> resultEntity = restTemplate.exchange(url, method, entity, bodyType);
        return resultEntity.getBody();
    }

    public RestTemplate getRestTemplate() {
        return restTemplate;
    }

}
