/**
 * Copyright (c) 2018 人人开源 All rights reserved.
 *
 * https://www.renren.io
 *
 * 版权所有，侵权必究！
 */

package com.ost.micro.modules.sys.controller;

import com.ost.micro.common.utils.Result;
import com.ost.micro.common.validator.AssertUtils;
import com.ost.micro.common.validator.ValidatorUtils;
import com.ost.micro.common.validator.group.AddGroup;
import com.ost.micro.common.validator.group.DefaultGroup;
import com.ost.micro.common.validator.group.UpdateGroup;
import com.ost.micro.core.utils.DateUtil;
import com.ost.micro.modules.sys.dto.SysDeptBizDTO;
import com.ost.micro.modules.sys.dto.SysDeptDTO;
import com.ost.micro.modules.sys.service.SysDeptService;
import com.ost.micro.common.annotation.LogOperation;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 部门管理
 * 
 * @author Mark sunlightcs@gmail.com
 */
@RestController
@RequestMapping("/sys/dept")
@Api(tags="部门管理")
public class SysDeptController {
	@Autowired
	private SysDeptService sysDeptService;

	@GetMapping("list")
	@ApiOperation("列表")
	@RequiresPermissions("sys:dept:list")
	public Result<List<SysDeptDTO>> list(@ApiIgnore @RequestParam Map<String, Object> params){
		List<SysDeptDTO> list = sysDeptService.list(params);

		return new Result<List<SysDeptDTO>>().ok(list);
	}

	@GetMapping("{id}")
	@ApiOperation("信息")
	@RequiresPermissions("sys:dept:info")
	public Result<SysDeptDTO> get(@PathVariable("id") String id){
		SysDeptDTO data = sysDeptService.get(id);

		return new Result<SysDeptDTO>().ok(data);
	}

	@PostMapping
	@ApiOperation("保存")
	@LogOperation("保存")
	@RequiresPermissions("sys:dept:save")
	public Result save(@RequestBody SysDeptDTO dto){

		dto.setDeptNum(DateUtil.getDateFormat(new Date(),"yyyyMMddHHmmss"));
		//效验数据
		ValidatorUtils.validateEntity(dto, AddGroup.class, DefaultGroup.class);

		sysDeptService.save(dto);

		return new Result();
	}

	@PutMapping
	@ApiOperation("修改")
	@LogOperation("修改")
	@RequiresPermissions("sys:dept:update")
	public Result update(@RequestBody SysDeptDTO dto){
		//效验数据
		ValidatorUtils.validateEntity(dto, UpdateGroup.class, DefaultGroup.class);

		sysDeptService.update(dto);

		return new Result();
	}

	@PutMapping("/biz/status")
	@ApiOperation("修改商家状态")
	@LogOperation("修改商家状态")
	@RequiresPermissions("sys:dept:update")
	public Result updateByBizId(@RequestBody SysDeptBizDTO dto){
		//效验数据
		ValidatorUtils.validateEntity(dto, UpdateGroup.class);
		 sysDeptService.updateByCondition(dto);
		 return new Result();
	}

	@DeleteMapping("{id}")
	@ApiOperation("删除")
	@LogOperation("删除")
	@RequiresPermissions("sys:dept:delete")
	public Result delete(@PathVariable("id") String id){
		//效验数据
		AssertUtils.isNull(id, "id");

		sysDeptService.delete(id);

		return new Result();
	}
	
}