/**
 * Copyright (c) 2018 人人开源 All rights reserved.
 *
 * https://www.renren.io
 *
 * 版权所有，侵权必究！
 */

package com.ost.micro.modules.security.controller;
import com.ost.micro.common.exception.ErrorCode;
import com.ost.micro.common.exception.RenException;
import com.ost.micro.common.utils.IpUtils;
import com.ost.micro.common.utils.Result;
import com.ost.micro.common.validator.AssertUtils;
import com.ost.micro.common.validator.ValidatorUtils;
import com.ost.micro.config.CustomFilterConfig;
import com.ost.micro.core.utils.GsonUtil;
import com.ost.micro.modules.log.entity.SysLogLoginEntity;
import com.ost.micro.modules.log.enums.LoginOperationEnum;
import com.ost.micro.modules.log.enums.LoginStatusEnum;
import com.ost.micro.modules.log.service.SysLogLoginService;
import com.ost.micro.modules.security.dto.LoginDTO;
import com.ost.micro.modules.security.password.PasswordUtils;
import com.ost.micro.modules.security.service.CaptchaService;
import com.ost.micro.modules.security.service.SysUserTokenService;
import com.ost.micro.modules.security.user.SecurityUser;
import com.ost.micro.modules.security.user.UserDetail;
import com.ost.micro.modules.sys.dto.SysUserExtDTO;
import com.ost.micro.modules.sys.enums.UserStatusEnum;
import com.ost.micro.modules.sys.service.SysUserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.netflix.zuul.filters.support.FilterConstants;
import org.springframework.http.HttpHeaders;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.*;

/**
 * 登录
 * 
 * @author Mark sunlightcs@gmail.com
 */
@RestController
@Api(tags="登录管理")
@Slf4j
public class LoginController {
	@Autowired
	private SysUserService sysUserService;
	@Autowired
	private SysUserTokenService sysUserTokenService;
	@Autowired
	private CaptchaService captchaService;
	@Autowired
	private SysLogLoginService sysLogLoginService;

	@Autowired
	private CustomFilterConfig customFilterConfig;

	@GetMapping("captcha")
	@ApiOperation(value = "验证码", produces="application/octet-stream")
	@ApiImplicitParam(paramType = "query", dataType="string", name = "uuid", required = true)
	public void captcha(HttpServletResponse response, String uuid)throws IOException {
		//uuid不能为空
		AssertUtils.isBlank(uuid, ErrorCode.IDENTIFIER_NOT_NULL);

		//生成图片验证码
		BufferedImage image = captchaService.create(uuid);

		response.setHeader("Cache-Control", "no-store, no-cache");
		response.setContentType("image/jpeg");
		ServletOutputStream out = response.getOutputStream();
		ImageIO.write(image, "jpg", out);
		out.close();
	}

	@PostMapping("login")
	@ApiOperation(value = "登录")
	public Result login(HttpServletRequest request, @RequestBody LoginDTO login) {
		log.info("开始登录>>>>>>>>>>>>>>>");
		String url = request.getScheme()+"://"+request.getServerName();
		log.info("请求url为:{}",url);
		//解析配置文件
		String[] domains = customFilterConfig.getDomains();
		List<String> domainsList = new LinkedList<>();
		if (domains.length > 0) {
			for (int i=0;i<domains.length ; i++) {
				String itemInfo = domains[i];
				if (StringUtils.isNotEmpty(itemInfo)) {
					String[] info = itemInfo.split(":");
					String domain = info[0];
					domainsList.add(domain);
				}
			}
		}

		log.info("解析配置文件域名信息结果为:{}", GsonUtil.toJson(domainsList,true));

		//效验数据
		ValidatorUtils.validateEntity(login);

		//验证码是否正确
		boolean flag = captchaService.validate(login.getUuid(), login.getCaptcha());
		if(!flag){
			return new Result().error(ErrorCode.CAPTCHA_ERROR,"验证码错误！");
		}

		//用户信息
		Map<String,Object> params = new HashMap<>();
		params.put("username",login.getUsername());
		params.put("bizId",login.getBizId());
		SysUserExtDTO user = sysUserService.getByUsername(params);
		SysLogLoginEntity sysLog = new SysLogLoginEntity();
		sysLog.setOperation(LoginOperationEnum.LOGIN.value());
		sysLog.setCreateDate(new Date());
		sysLog.setIp(IpUtils.getIpAddr(request));
		sysLog.setUserAgent(request.getHeader(HttpHeaders.USER_AGENT));
		//sysLog.setIp(IpUtils.getIpAddr(request));
		// 判断是商户登录还是商家登录,通过url去判断
		String backstage = "";
		String backmch = "";

		if (domainsList.size() == 1 ) {
			backstage = StringUtils.isEmpty(domainsList.get(0)) ?  "backstage":domainsList.get(0);
			backmch = "backmch";
		}
		if (domainsList.size() == 2 ) {
			backstage = StringUtils.isEmpty(domainsList.get(0)) ?  "backstage":domainsList.get(0);
			backmch = StringUtils.isEmpty(domainsList.get(1)) ?  "backmch":domainsList.get(1);
		}
		log.info("backstage = {},backmch = {}",backstage,backmch);
		if (url.contains(backmch)) {//商户后台登录
			if ("admin".equals(login.getUsername())) {
				sysLog.setStatus(LoginStatusEnum.FAIL.value());
				sysLog.setCreaterName(login.getUsername());
				sysLogLoginService.save(sysLog);
				log.info("[admin账号]不能登录商家后台");
				throw new RenException("[admin账号]不能登录商家后台",ErrorCode.ACCOUNT_ERROR);
			}
			if (StringUtils.isEmpty(login.getBizId())) {
				log.info("商家号为空或者错误");
				sysLog.setStatus(LoginStatusEnum.FAIL.value());
				sysLog.setCreaterName(login.getUsername());
				sysLogLoginService.save(sysLog);
				throw new RenException("商家号为空或者错误",ErrorCode.BIZID_ERROR_OR_NULL);
			}
		}else if (url.contains(backstage)) {//运营商后台登录
			if (null != user) {
				String bizId = user.getBizId();
				if (StringUtils.isNotEmpty(bizId)) {
					log.info("商家账号不能登录运营商后台");
					sysLog.setStatus(LoginStatusEnum.FAIL.value());
					sysLog.setCreaterName(login.getUsername());
					sysLogLoginService.save(sysLog);
					throw new RenException("商家账号不能登录运营商后台",ErrorCode.ACCOUNT_ERROR);
				}
			}
		}

		//用户不存在
		if(user == null){
			sysLog.setStatus(LoginStatusEnum.FAIL.value());
			sysLog.setCreaterName(login.getUsername());
			sysLogLoginService.save(sysLog);
			throw new RenException("用户不存在",ErrorCode.ACCOUNT_PASSWORD_ERROR);
		}
		//密码错误
		if(!PasswordUtils.matches(login.getPassword(), user.getPassword())){
			sysLog.setStatus(LoginStatusEnum.FAIL.value());
			sysLog.setCreater(user.getId());
			sysLog.setCreaterName(user.getUsername());
			sysLogLoginService.save(sysLog);
			throw new RenException("密码错误",ErrorCode.ACCOUNT_PASSWORD_ERROR);
		}
		//账号停用
		if(user.getStatus() == UserStatusEnum.DISABLE.value()){
			sysLog.setStatus(LoginStatusEnum.LOCK.value());
			sysLog.setCreater(user.getId());
			sysLog.setCreaterName(user.getUsername());
			sysLogLoginService.save(sysLog);
			throw new RenException("账号停用",ErrorCode.ACCOUNT_DISABLE);
		}

		//商家账号停用
		if(null != user.getDeptStatus() && user.getDeptStatus() == UserStatusEnum.DISABLE.value()){
			sysLog.setStatus(LoginStatusEnum.LOCK.value());
			sysLog.setCreater(user.getId());
			sysLog.setCreaterName(user.getUsername());
			sysLogLoginService.save(sysLog);
			throw new RenException("商家账号停用",ErrorCode.ACCOUNT_DISABLE);
		}


		//登录成功
		sysLog.setStatus(LoginStatusEnum.SUCCESS.value());
		sysLog.setCreater(user.getId());
		sysLog.setCreaterName(user.getUsername());
		sysLogLoginService.save(sysLog);
		Result result = sysUserTokenService.createToken(user.getId());
		log.info("登录结束>>>>>>>>>>>>>>>result:{}",result);
		return  result;
	}

	@PostMapping("logout")
	@ApiOperation(value = "退出")
	public Result logout(HttpServletRequest request) {
		UserDetail user = SecurityUser.getUser();

		//退出
		sysUserTokenService.logout(user.getId());

		//用户信息
		SysLogLoginEntity log = new SysLogLoginEntity();
		log.setOperation(LoginOperationEnum.LOGOUT.value());
		log.setCreateDate(new Date());
		log.setIp(IpUtils.getIpAddr(request));
		log.setUserAgent(request.getHeader(HttpHeaders.USER_AGENT));
		log.setIp(IpUtils.getIpAddr(request));
		log.setStatus(LoginStatusEnum.SUCCESS.value());
		log.setCreater(user.getId());
		log.setCreaterName(user.getUsername());
		sysLogLoginService.save(log);

		return new Result();
	}
	
}