/**
 * Copyright (c) 2018 人人开源 All rights reserved.
 *
 * https://www.renren.io
 *
 * 版权所有，侵权必究！
 */

package com.ost.micro.modules.sys.controller;

import com.ost.micro.common.constant.Constant;
import com.ost.micro.common.exception.ErrorCode;
import com.ost.micro.common.page.PageData;
import com.ost.micro.common.utils.ConvertUtils;
import com.ost.micro.common.utils.Result;
import com.ost.micro.common.validator.AssertUtils;
import com.ost.micro.common.validator.ValidatorUtils;
import com.ost.micro.common.validator.group.AddGroup;
import com.ost.micro.common.validator.group.DefaultGroup;
import com.ost.micro.common.validator.group.UpdateGroup;
import com.ost.micro.core.utils.DateUtil;
import com.ost.micro.modules.security.password.PasswordUtils;
import com.ost.micro.modules.security.user.SecurityUser;
import com.ost.micro.modules.security.user.UserDetail;
import com.ost.micro.modules.sys.dto.PasswordDTO;
import com.ost.micro.modules.sys.dto.SysDeptDTO;
import com.ost.micro.modules.sys.dto.SysUserDTO;
import com.ost.micro.modules.sys.dto.SysUserExtDTO;
import com.ost.micro.modules.sys.excel.SysUserExcel;
import com.ost.micro.modules.sys.service.SysDeptService;
import com.ost.micro.modules.sys.service.SysRoleUserService;
import com.ost.micro.modules.sys.service.SysUserService;
import com.ost.micro.securitycommon.annotation.LogOperation;
import com.ost.micro.securitycommon.utils.ExcelUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 用户管理
 * 
 * @author Mark sunlightcs@gmail.com
 */
@RestController
@RequestMapping("/sys/user")
@Api(tags="用户管理")
public class SysUserController {
	@Autowired
	private SysUserService sysUserService;
	@Autowired
	private SysRoleUserService sysRoleUserService;
	@Autowired
	private SysDeptService sysDeptService;

	@GetMapping("page")
	@ApiOperation("分页")
	@ApiImplicitParams({
		@ApiImplicitParam(name = Constant.PAGE, value = "当前页码，从1开始", paramType = "query", required = true, dataType="int") ,
		@ApiImplicitParam(name = Constant.LIMIT, value = "每页显示记录数", paramType = "query",required = true, dataType="int") ,
		@ApiImplicitParam(name = Constant.ORDER_FIELD, value = "排序字段", paramType = "query", dataType="String") ,
		@ApiImplicitParam(name = Constant.ORDER, value = "排序方式，可选值(asc、desc)", paramType = "query", dataType="String") ,
		@ApiImplicitParam(name = "username", value = "用户名", paramType = "query", dataType="String")
	})
	@RequiresPermissions("sys:user:page")
	public Result<PageData<SysUserExtDTO>> page(@ApiIgnore @RequestParam Map<String, Object> params){
		PageData<SysUserExtDTO> page = sysUserService.page(params);
		if (null != page && !CollectionUtils.isEmpty(page.getList())) {
			page.getList().forEach(e->{
				//用户角色列表
				List<String> roleIdList = sysRoleUserService.getRoleIdList(e.getId());
				e.setRoleIdList(roleIdList);
			});
		}
		return new Result<PageData<SysUserExtDTO>>().ok(page);
	}

	@GetMapping("{id}")
	@ApiOperation("信息")
	@RequiresPermissions("sys:user:info")
	public Result<SysUserDTO> get(@PathVariable("id") String id){
		SysUserDTO data = sysUserService.get(id);
		SysUserExtDTO sysUserExtDTO = new SysUserExtDTO();
		BeanUtils.copyProperties(data,sysUserExtDTO);
		//用户角色列表
		List<String> roleIdList = sysRoleUserService.getRoleIdList(id);
		data.setRoleIdList(roleIdList);
		sysUserExtDTO.setRoleIdList(roleIdList);
		if (null != data) {
			SysDeptDTO deptDTO = sysDeptService.get(data.getDeptId());
			sysUserExtDTO.setDeptName(deptDTO.getName());
		}

		return new Result<SysUserDTO>().ok(sysUserExtDTO);
	}

	@GetMapping("info")
	@ApiOperation("登录用户信息")
	//@RequiresPermissions("sys:user:info")
	public Result<SysUserDTO> info(){
		SysUserDTO data = ConvertUtils.sourceToTarget(SecurityUser.getUser(), SysUserDTO.class);
		return new Result<SysUserDTO>().ok(data);
	}

	@PutMapping("password")
	@ApiOperation("修改密码")
	@LogOperation("修改密码")
	public Result password(@RequestBody PasswordDTO dto){
		//效验数据
		ValidatorUtils.validateEntity(dto);

		UserDetail user = SecurityUser.getUser();

		//原密码不正确
		if(!PasswordUtils.matches(dto.getPassword(), user.getPassword())){
			return new Result().error(ErrorCode.PASSWORD_ERROR);
		}

		sysUserService.updatePassword(user.getId(), dto.getNewPassword());

		return new Result();
	}

	@PostMapping
	@ApiOperation("保存")
	@LogOperation("保存")
	@RequiresPermissions("sys:user:save")
	public Result save(@RequestBody SysUserDTO dto){
		//效验数据
		dto.setUserNum(DateUtil.getDateFormat(new Date(),"yyyyMMddHHmmss"));
		ValidatorUtils.validateEntity(dto, AddGroup.class, DefaultGroup.class);
		sysUserService.save(dto);

		return new Result();
	}

	@PutMapping
	@ApiOperation("修改")
	@LogOperation("修改")
	@RequiresPermissions("sys:user:update")
	public Result update(@RequestBody SysUserDTO dto){
		//效验数据
		ValidatorUtils.validateEntity(dto, UpdateGroup.class, DefaultGroup.class);

		sysUserService.update(dto);

		return new Result();
	}

	@DeleteMapping
	@ApiOperation("删除")
	@LogOperation("删除")
	@RequiresPermissions("sys:user:delete")
	public Result delete(@RequestBody String[] ids){
		if(ArrayUtils.contains(ids, 1L)){
			return new Result().error("系统管理员不能删除");
		}

		//效验数据
		AssertUtils.isArrayEmpty(ids, "id");

		sysUserService.deleteBatchIds(Arrays.asList(ids));

		return new Result();
	}

	@GetMapping("export")
	@ApiOperation("导出")
	@LogOperation("导出")
	@RequiresPermissions("sys:user:export")
	@ApiImplicitParam(name = "username", value = "用户名", paramType = "query", dataType="String")
	public void export(@ApiIgnore @RequestParam Map<String, Object> params, HttpServletResponse response) throws Exception {
		List<SysUserDTO> list = sysUserService.list(params);

		ExcelUtils.exportExcelToTarget(response, null, list, SysUserExcel.class);
	}
}