package com.ost.micro.provider.controller;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.ost.micro.common.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.modules.cardservice.dto.order.response.OrderResDto;
import com.ost.micro.modules.cardservice.dto.order.response.OrderStatisticsResDto;
import com.ost.micro.modules.cardservice.excel.OrderExcelDto;
import com.ost.micro.modules.cardservice.excel.OrderStatisticsExcelDto;
import com.ost.micro.core.utils.DateUtil;
import com.ost.micro.core.utils.ExcelUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/provider/card/statics")
@Api(tags = "card交易统计")
@Slf4j
public class CardOrderStaticsController {

    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    @GetMapping("")
    @ApiOperation("交易统计查询")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_list", value = "平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "payment_type", value = "支付方式 0-银行卡 1-固码汇旺财 3-PDD", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "passage_name", value = "通道名称", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "begin_time", value = "开始时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "end_time", value = "结束时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse<List<OrderStatisticsResDto>> search(
            @RequestParam(required = false, name = "biz_id") Long bizId,
            @RequestParam(required = false, name = "payment_type") Integer paymentType,
            @RequestParam(required = false, name = "passage_name") String passageName,
            @RequestParam(required = false, name = "biz_id_list") List<Long> bizIdList,
            @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
            @RequestParam(required = false, name = "begin_time") Long beginTime,
            @RequestParam(required = false, name = "end_time") Long endTime,
            @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
            @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
            @RequestParam(name = "page_index") Integer pageIndex,
            @RequestParam(name = "page_size") Integer pageSize) {
        String staticUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "api/provider/card/statics";
        Map<String, Object> params = new HashMap<>();
        params.put("bizId", bizId);
        if (null != paymentType) {
            params.put("paymentType", paymentType);
        }
        params.put("passageName", passageName);
        if (!CollectionUtils.isEmpty(bizIdList)) {
            bizIds = bizIdList;
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("beginTime", beginTime);
        params.put("endTime", endTime);
        params.put("endTime", endTime);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        return paymentServiceClient.get(staticUrl, params);
    }

    /**
     * 交易明细
     *
     * @return
     */
    @GetMapping("/item")
    @ApiOperation("交易统计明细")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "passage_id", value = "通道id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "payment_type", value = "支付方式", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "begin_time", value = "开始时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "end_time", value = "结束时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse tradePage(@RequestParam(required = false, name = "keyword") String keyword,
                                  @RequestParam(name = "biz_id") Long bizId,
                                  @RequestParam(required = false, name = "payment_type") Integer paymentType,
                                  @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                  @RequestParam(name = "passage_id") Long passageId,
                                  @RequestParam(required = false, name = "begin_time") Long beginTime,
                                  @RequestParam(required = false, name = "end_time") Long endTime,
                                  @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                                  @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                  @RequestParam(name = "page_index") Integer pageIndex,
                                  @RequestParam(name = "page_size") Integer pageSize) {
        String staticUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/statics/item";
        Map<String, Object> params = new HashMap<>();
        params.put("keyword", keyword);
        params.put("bizId", bizId);
        if (null != paymentType) {
            params.put("paymentType", paymentType);
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("passageId", passageId);
        params.put("beginTime", beginTime);
        params.put("endTime", endTime);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        return paymentServiceClient.get(staticUrl, params);
    }

    @GetMapping("/export")
    @ApiOperation("导出交易统计")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_list", value = "平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "payment_type", value = "支付方式", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "begin_time", value = "开始时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "end_time", value = "结束时间", required = false, dataType = "long")})
    public void exportTradeStatic(@RequestParam(required = false, name = "biz_id") Long bizId,
                                  @RequestParam(required = false, name = "biz_id_list") List<Long> bizIdList,
                                  @RequestParam(required = false, name = "payment_type") Integer paymentType,
                                  @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                  @RequestParam(required = false, name = "begin_time") Long beginTime,
                                  @RequestParam(required = false, name = "end_time") Long endTime,
                                  @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                                  @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                  HttpServletResponse response) {
        String staticUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/statics/export";
        Map<String, Object> params = new HashMap<>();
        params.put("bizId", bizId);
        if (!CollectionUtils.isEmpty(bizIdList)) {
            bizIds = bizIdList;
        }
        if (null != paymentType) {
            params.put("paymentType", paymentType);
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("beginTime", beginTime);
        params.put("endTime", endTime);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);
        DataResponse dataResponse = paymentServiceClient.get(staticUrl, params);
        if (null != dataResponse && null != dataResponse.getData()) {
            Object obj = dataResponse.getData().getList();
            Gson gson = new Gson();
            if (null != obj && obj instanceof List) {
                List<OrderStatisticsResDto> dataList = gson.fromJson(gson.toJson(obj), new TypeToken<List<OrderStatisticsResDto>>() {
                }.getType());
                try {
                    ExcelUtils.exportExcelToTarget(response, "交易统计_" + DateUtil.format(new Date(), "yyyyMMddHHmmss"), dataList, OrderStatisticsExcelDto.class);
                } catch (Exception e) {
                    log.error("export 导出交易流水 error", e);
                }
            }
        }

    }

    @GetMapping("/export/item")
    @ApiOperation("导出交易统计明细")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "payment_type", value = "支付方式", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "passage_id", value = "通道id", required = true, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "begin_time", value = "开始时间", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "end_time", value = "结束时间", required = false, dataType = "long")})
    public void exportTradePage(@RequestParam(required = false) String keyword,
                                @RequestParam(name = "biz_id") Long bizId,
                                @RequestParam(required = false, name = "payment_type") Integer paymentType,
                                @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                @RequestParam(name = "passage_id") Long passageId,
                                @RequestParam(required = false, name = "begin_time") Long beginTime,
                                @RequestParam(required = false, name = "end_time") Long endTime,
                                @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                                @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                HttpServletResponse response) {
        String staticUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/statics/export/item";
        Map<String, Object> params = new HashMap<>();
        params.put("keyword", keyword);
        params.put("bizId", bizId);
        if (null != paymentType) {
            params.put("paymentType", paymentType);
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("passageId", passageId);
        params.put("beginTime", beginTime);
        params.put("endTime", endTime);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);
        DataResponse dataResponse = paymentServiceClient.get(staticUrl, params);
        if (null != dataResponse && null != dataResponse.getData()) {
            Object obj = dataResponse.getData().getList();
            Gson gson = new Gson();
            if (null != obj && obj instanceof List) {
                List<OrderResDto> dataList = gson.fromJson(gson.toJson(obj), new TypeToken<List<OrderResDto>>() {
                }.getType());
                try {
                    ExcelUtils.exportExcelToTarget(response, "统计明细_" + DateUtil.format(new Date(), "yyyyMMddHHmmss"), dataList, OrderExcelDto.class);
                } catch (Exception e) {
                    log.error("export 导出交易统计明细 error", e);
                }
            }
        }
    }
}
