package com.ost.micro.provider.controller;

import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.modules.cardservice.dto.order.request.OrderReqDto;
import com.ost.micro.core.pay.modules.cardservice.dto.order.response.OrderResDto;
import com.ost.micro.core.pay.modules.cardservice.excel.OrderExcelDto;
import com.ost.micro.core.utils.DateUtil;
import com.ost.micro.core.utils.ExcelUtils;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Yubo
 */
@RestController
@RequestMapping("/provider/card/order")
@Api(tags = "card交易明细接口")
@Slf4j
public class CardOrderController {

    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    @GetMapping("")
    @ApiOperation("查询交易明细列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "客户姓名", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "remark", value = "备注", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "amount", value = "金额", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_list", value = "平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "权限平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
            @ApiImplicitParam(paramType = "query", name = "status", value = "状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "from_date", value = "创建起始日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "to_date", value = "创建结束日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse<List<OrderResDto>> search(@RequestParam(required = false) String keyword,
                                                  @RequestParam(required = false) String name,
                                                  @RequestParam(required = false) String remark,
                                                  @RequestParam(required = false) String amount,
                                                  @RequestParam(required = false, name = "biz_id") Long bizId,
                                                  @RequestParam(required = false, name = "biz_id_list") List<Long> bizIdList,
                                                  @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                                  @RequestParam(required = false) Integer status,
                                                  @RequestParam(required = false, name = "from_date") Long fromDate,
                                                  @RequestParam(required = false, name = "to_date") Long toDate,
                                                  @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                                                  @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                                                  @RequestParam(name = "page_index") Integer pageIndex,
                                                  @RequestParam(name = "page_size") Integer pageSize) {
        String orderUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/order";
        Map<String, Object> params = new HashMap<>();
        params.put("keyword", keyword);
        params.put("name", name);
        params.put("remark", remark);
        params.put("amount", amount);
        params.put("bizId", bizId);
        if (!CollectionUtils.isEmpty(bizIdList)) {
            bizIds = bizIdList;
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("status", status);
        params.put("fromDate", fromDate);
        params.put("toDate", toDate);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);

        return paymentServiceClient.get(orderUrl, params);
    }

    @PutMapping("")
    @ApiOperation("编辑订单")
    @DataToUnderline()
    public DataResponse edit(@RequestBody OrderReqDto dto) {
        String orderUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/order";
        return paymentServiceClient.putForEntity(orderUrl, dto);
    }

    @GetMapping("/{id}")
    @ApiOperation("获取交易详情")
    @ApiImplicitParam(paramType = "path", name = "id", value = "订单id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse getTradeDetail(@PathVariable Long id) {
        String orderUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/order";
        return paymentServiceClient.get(orderUrl, id);
    }

    @GetMapping("/export")
    @ApiOperation("导出交易流水")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "name", value = "客户姓名", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "remark", value = "备注", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "amount", value = "金额", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "biz_id", value = "平台id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_id_list", value = "平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "type", value = "交易类型", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "status", value = "状态", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "from_date", value = "起始日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "to_date", value = "结束日期", required = false, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "current_biz_id", value = "当前登录的商家id", required = false, dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "is_biz_id_from_permission", value = "bizId是否是权限系统中", required = false, dataType = "boolean"),
    })
    public void export(@RequestParam(required = false) String keyword,
                       @RequestParam(required = false) String name,
                       @RequestParam(required = false) String remark,
                       @RequestParam(required = false) String amount,
                       @RequestParam(required = false, name = "biz_id") Long bizId,
                       @RequestParam(required = false, name = "biz_id_list") List<Long> bizIdList,
                       @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                       @RequestParam(required = false) Integer status,
                       @RequestParam(required = false, name = "from_date") Long fromDate,
                       @RequestParam(required = false, name = "to_date") Long toDate,
                       @RequestParam(required = false, name = "begin_time") Long beginTime,
                       @RequestParam(required = false, name = "end_time") Long endTime,
                       @RequestParam(required = false, name = "current_biz_id") Long currentBizId,
                       @RequestParam(required = false, name = "is_biz_id_from_permission") Boolean isBizIdFromPermission,
                       HttpServletResponse response) {

        String orderUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/api/provider/card/order/export";
        Map<String, Object> params = new HashMap<>();
        params.put("keyword", keyword);
        params.put("name", name);
        params.put("remark", remark);
        params.put("amount", amount);
        params.put("bizId", bizId);
        if (!CollectionUtils.isEmpty(bizIdList)) {
            bizIds = bizIdList;
        }
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("status", status);
        params.put("fromDate", fromDate);
        params.put("toDate", toDate);
        params.put("beginTime", beginTime);
        params.put("endTime", endTime);
        params.put("currentBizId", currentBizId);
        params.put("isBizIdFromPermission", isBizIdFromPermission);

        DataResponse dataResponse = paymentServiceClient.get(orderUrl, params);
        if (null != dataResponse && null != dataResponse.getData()) {
            Object obj = dataResponse.getData().getList();
            Gson gson = new Gson();
            if (null != obj && obj instanceof List) {
                List<OrderResDto> dataList = gson.fromJson(gson.toJson(obj), new TypeToken<List<OrderResDto>>() {
                }.getType());
                try {
                    ExcelUtils.exportExcelToTarget(response, "交易明细_" + DateUtil.format(new Date(), "yyyyMMddHHmmss"), dataList, OrderExcelDto.class);
                } catch (Exception e) {
                    log.error("export 导出交易流水 error", e);
                }
            }
        }
    }

}
