package com.ost.micro.provider.controller;
import com.ost.micro.core.aop.DataToUnderline;
import com.ost.micro.core.context.model.response.DataResponse;
import com.ost.micro.core.pay.modules.cardservice.dto.biz.request.BizParamsReqDto;
import com.ost.micro.core.pay.modules.cardservice.dto.biz.request.BizReqDto;
import com.ost.micro.core.pay.modules.cardservice.dto.biz.response.BizResDto;
import com.ost.micro.provider.common.PaymentServiceClient;
import com.ost.micro.provider.common.ServiceUrl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.consul.discovery.ConsulDiscoveryClient;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 商家管理表
 *
 * @author rain
 * @email rain.guo@bytehug.com
 * @date 2019-08-13 16:42:06
 */
@RestController
@RequestMapping("provider/card/biz")
@Api(tags = "card商家管理表接口")
@Slf4j
public class CardBizController {
    @Autowired
    private PaymentServiceClient paymentServiceClient;
    @Autowired
    private ConsulDiscoveryClient consulDiscoveryClient;
    @Autowired
    private ServiceUrl serviceUrl;

    @GetMapping("")
    @ApiOperation("分页列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "keyword", value = "关键字", required = false, dataType = "String"),
            @ApiImplicitParam(paramType = "query", name = "enable", value = "0禁用1启用", required = false, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long"),
            @ApiImplicitParam(paramType = "query", name = "page_index", value = "页码", required = true, dataType = "int"),
            @ApiImplicitParam(paramType = "query", name = "page_size", value = "每页数码条数", required = true, dataType = "int")})
    @DataToUnderline()
    public DataResponse<List<BizResDto>> search(@RequestParam(required = false, name = "keyword") String keyword,
                                                @RequestParam(required = false, name = "enable") Integer enable,
                                                @RequestParam(required = false, name = "biz_ids") List<Long> bizIds,
                                                @RequestParam(name = "page_index") Integer pageIndex,
                                                @RequestParam(name = "page_size") Integer pageSize) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz";
        Map params = new HashMap();
        params.put("keyword", keyword);
        params.put("enable", enable);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        params.put("pageIndex", pageIndex);
        params.put("pageSize", pageSize);

        return paymentServiceClient.get(bizUrl, params);
    }

    @GetMapping("list")
    @ApiOperation("列表")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", name = "bank_id", value = "银行id", dataType = "long"),
            @ApiImplicitParam(paramType = "query", name = "biz_ids", value = "所属平台集合", required = false, allowMultiple = true, dataType = "Long")})
    @DataToUnderline()
    public DataResponse list(@RequestParam(required = false, name = "bank_id") Long bankId,
                             @RequestParam(required = false, name = "biz_ids") List<Long> bizIds) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz/list";
        Map params = new HashMap();
        params.put("bankId", bankId);
        params.put("bizIds", !CollectionUtils.isEmpty(bizIds) ? StringUtils.strip(bizIds.toString(), "[]") : null);
        return paymentServiceClient.get(bizUrl, params);
    }


    @GetMapping("/{id}")
    @ApiOperation("查看")
    @ApiImplicitParam(paramType = "path", name = "id", value = "id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse<BizResDto> info(@PathVariable("id") Long id) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz";
        return paymentServiceClient.get(bizUrl, id);
    }

    @PostMapping("")
    @ApiOperation("新增")
    @DataToUnderline()
    public DataResponse add(@RequestBody BizReqDto dto) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz";
        return paymentServiceClient.post(bizUrl, dto);

    }

    @PutMapping("")
    @ApiOperation("修改")
    @DataToUnderline()
    public DataResponse<BizResDto> update(@RequestBody BizReqDto dto) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz";
        return paymentServiceClient.putForEntity(bizUrl, dto);
    }

    @DeleteMapping("{id}")
    @ApiOperation("删除")
    @ApiImplicitParam(paramType = "path", name = "id", value = "id", required = true, dataType = "long")
    @DataToUnderline()
    public DataResponse delete(@PathVariable Long id) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz/" + id;
        ResponseEntity<DataResponse> resp = paymentServiceClient.getRestTemplate()
                .exchange(bizUrl, HttpMethod.DELETE, null,
                        DataResponse.class, id);
        return resp.getBody();

    }

    @PutMapping("/params/edit")
    @ApiOperation("修改交易参数")
    @DataToUnderline()
    public DataResponse<BizResDto> editParams(@RequestBody BizParamsReqDto dto) {
        String bizUrl = serviceUrl.getAsServiceUrl(consulDiscoveryClient) + "/card/biz/params/edit";
        return paymentServiceClient.putForEntity(bizUrl, dto);
    }
}
